<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Hotel extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'hotels';
    protected $fillable = [
        'destination_id',
        'name',
        'phone',
        'front_image',
        'guest',
        'price',
        'type',
        'service_charge',
        'tax_fees',
        'room',
        'address',
        'latitude',
        'longitude',
        'license_document',
        'description',
        'status',
        'is_recommended'
    ];

    public function additionalInfos()
    {
        return $this->hasMany(Additional_Info::class, 'hotel_id');
    }
    
    public function cancelBookings()
    {
        return $this->hasMany(CancelBooking::class);
    }
    
    public function recentlyViewedHotels()
    {
        return $this->hasMany(RecentlyViewedHotel::class);
    }
    
    public function destination()
    {
        return $this->belongsTo(Destination::class, 'destination_id', 'id');
    }

    public function cancellationPolicies()
    {
        return $this->hasMany(Cancellation_Policy::class, 'hotel_id');
    }
    
    public function offdays()
    {
        return $this->hasMany(Hotel_Off_Days::class, 'hotel_id');
    }
    
    public function specification()
    {
        return $this->hasMany(Specification::class, 'hotel_id');
    }

    public function hotelAminities()
    {
        return $this->hasMany(Hotel_Aminities::class, 'hotel_id');
    }
    
    public function coupons()
    {
        return $this->belongsToMany(Coupon::class, 'coupon_hotels', 'hotel_id', 'coupon_id');
    }
    
    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function hotelPackages()
    {
        return $this->hasMany(Hotel_Packages::class, 'hotel_id');
    }
    
    // public function hotelFeatures()
    // {
    //     return $this->hasMany(HotelFeature::class, 'hotel_id');
    // }
    
    public function hotelRules()
    {
        return $this->hasMany(Hotel_Rules::class, 'hotel_id');
    }

    public function hotelImages()
    {
        return $this->hasMany(Hotel_Images::class, 'hotel_id');
    }

    // public function category()
    // {
    //     return $this->belongsTo(Category::class, 'category_id');
    // }
    
    public function vendor()
    {
        return $this->belongsTo(Vendor::class, 'vendor_id');
    }
 
    public function aminities()
    {
        return $this->belongsToMany(Aminity::class, 'hotel_aminities', 'hotel_id', 'aminities_id');
    }
    
    /**
     * Fetch list of from here
    **/
    public static function getLists($search)
    {
        try {
            $obj = new self;

            $pagination = (isset($search['perpage'])) ? $search['perpage'] : config('constant.pagination');

            if (isset($search['name']) && !empty($search['name'])) {
                $obj = $obj->where('name', 'like', '%' . trim($search['name']) . '%');
            }

            if (isset($search['status']) && $search['status'] !== '') {
                $obj = $obj->where('status', $search['status']);
            }

            if (isset($search['phone']) && $search['phone'] !== '') {
                $obj = $obj->where('phone', $search['phone']);
            }

            if (isset($search['destination_id']) && $search['destination_id'] !== '') {
                $obj = $obj->where('destination_id', $search['destination_id']);
            }

            $data = $obj->with(['destination'])->where('add_type','Admin')->orderBy('id', 'DESC')->paginate($pagination)->appends('perpage', $pagination);

            return $data;
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
        }
    }
    
    /**
     * Fetch list of from here
    **/
    public static function getVendorLists($search)
    {
        try {
            $obj = new self;

            $pagination = (isset($search['perpage'])) ? $search['perpage'] : config('constant.pagination');

            if (isset($search['name']) && !empty($search['name'])) {
                $obj = $obj->where('name', 'like', '%' . trim($search['name']) . '%');
            }

            if (isset($search['status']) && $search['status'] !== '') {
                $obj = $obj->where('status', $search['status']);
            }

            if (isset($search['phone']) && $search['phone'] !== '') {
                $obj = $obj->where('phone', $search['phone']);
            }

            if (isset($search['destination_id']) && $search['destination_id'] !== '') {
                $obj = $obj->where('destination_id', $search['destination_id']);
            }
            $data = $obj->with(['destination','vendor'])->where('add_type','Vendor')->orderBy('id', 'DESC')->paginate($pagination)->appends('perpage', $pagination);

            return $data;
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
        }
    }
    
    /**
     * Add or update
    **/
    public static function addUpdate($data, $id = 0)
    {
        try {
            $obj = new self;
            $aminities_id_arr = $data['aminities_id'] ?? [];
            $additional_title_arr = $data['additional_title'] ?? [];
            $cancellation_policy_title_arr = $data['cancellation_policy_title'] ?? [];
            $specification_title_arr = $data['specification_title'] ?? [];
            $images_arr = $data['images'] ?? [];
            $feature_id_arr = $data['feature_id'] ?? [];
            $rules_id_arr = $data['rules_id'] ?? [];
            $off_days_arr = $data['off_days'] ?? [];

            unset($data['_token']);
            unset($data['aminities_id']);
            unset($data['additional_title']);
            unset($data['cancellation_policy_title']);
            unset($data['specification_title']);
            unset($data['feature_id']);
            unset($data['images']);
            unset($data['rules_id']);
            unset($data['off_days']);
    
            // Handle image uploads
            if (isset($data['front_image']) && $data['front_image'] != '') {
                $uploadImage = \Helpers::uploadFiles($data['front_image'], 'hotel/front_image');
                if ($uploadImage['status'] == true) {
                    $data['front_image'] = $uploadImage['file_name'];
                }
            }
            if (isset($data['license_document']) && $data['license_document'] != '') {
                $uploadImage = \Helpers::uploadFiles($data['license_document'], 'hotel/license_documents');
                if ($uploadImage['status'] == true) {
                    $data['license_document'] = $uploadImage['file_name'];
                }
            }
    
            if ($id == 0) {
                $data['created_at'] = date('Y-m-d H:i:s');
                $hotel_id = $obj->insertGetId($data);
                
                // Handle amenities
                if(isset($aminities_id_arr) && $aminities_id_arr!=''){
                        foreach ($aminities_id_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'aminities_id' => $value
                            );
                            Hotel_Aminities::create($data);
                        }
                }
    
                // Handle additional info
                if(isset($additional_title_arr) && $additional_title_arr!=''){
                        foreach ($additional_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'title' => $value
                            );
                            Additional_Info::create($data);
                        }
                }
    
                // Handle cancellation policies
                if(isset($cancellation_policy_title_arr) && $cancellation_policy_title_arr!=''){
                        foreach ($cancellation_policy_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'title' => $value
                            );
                            Cancellation_Policy::create($data);
                        }
                }
    
                // Handle specifications
                if(isset($specification_title_arr) && $specification_title_arr!=''){
                        foreach ($specification_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'title' => $value
                            );
                            Specification::create($data);
                        }
                }
    
                // Handle image uploads
                if (isset($images_arr) && !empty($images_arr)) {
                            foreach ($images_arr as $image) {
                        \Log::info('Uploading image: ', ['image' => $image->getClientOriginalName()]);
        
                        $uploadImage = \Helpers::multipleUploadFiles($image, 'hotel/images');
                        if ($uploadImage['status'] == true) {
                            Hotel_Images::create([
                                'hotel_id' => $hotel_id,
                                'images' => $uploadImage['file_name'],
                            ]);
                        } else {
                            \Log::error('Image upload failed: ', ['error' => $uploadImage['message']]);
                        }
                    }
                }
                
                // // Handle additional info
                // if(isset($feature_id_arr) && $feature_id_arr!=''){
                //         foreach ($feature_id_arr as $feature_id) {
                //             $data = array(
                //                 'hotel_id'=> $hotel_id,
                //                 'feature_id' => $feature_id
                //             );
                //             HotelFeature::create($data);
                //         }
                // }
                
                // Insert rules
                if(isset($rules_id_arr) && $rules_id_arr!=''){
                    foreach ($rules_id_arr as $rules_id) {
                        $data = array(
                            'hotel_id'=> $hotel_id,
                            'rules_id' => $rules_id
                        );
                        Hotel_Rules::create($data);
                    }
                }
                
                // Insert off days
                if(isset($off_days_arr) && $off_days_arr!=''){
                    foreach ($off_days_arr as $offDay) {
                        $data = array(
                            'hotel_id'=> $hotel_id,
                            'off_days' => $offDay
                        );
                        Hotel_Off_Days::create($data);
                    }
                }
                
                return ['status' => true, 'message' => __('lang.admin_data_add_msg')];
            } else {
                $data['updated_at'] = date('Y-m-d H:i:s');
                $obj->where('id',$id)->update($data);
                
                // Handle amenities
                Hotel_Aminities::where('hotel_id',$id)->delete();
                if(isset($aminities_id_arr) && $aminities_id_arr!=''){
                        foreach ($aminities_id_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $id,
                                'aminities_id' => $value
                            );
                            Hotel_Aminities::create($data);
                        }
                }
                
                // Handle additional info
                Additional_Info::where('hotel_id',$id)->delete();
                if(isset($additional_title_arr) && $additional_title_arr!=''){
                        foreach ($additional_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $id,
                                'title' => $value
                            );
                            Additional_Info::create($data);
                        }
                }
    
                // Handle cancellation policies
                Cancellation_Policy::where('hotel_id',$id)->delete();
                if(isset($cancellation_policy_title_arr) && $cancellation_policy_title_arr!=''){
                        foreach ($cancellation_policy_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $id,
                                'title' => $value
                            );
                            Cancellation_Policy::create($data);
                        }
                }
    
                // Handle specifications
                Specification::where('hotel_id',$id)->delete();
                if(isset($specification_title_arr) && $specification_title_arr!=''){
                        foreach ($specification_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $id,
                                'title' => $value
                            );
                            Specification::create($data);
                        }
                }
    
                // Handle image uploads
                if (isset($images_arr) && !empty($images_arr)) {
                            foreach ($images_arr as $image) {
                        \Log::info('Uploading image: ', ['image' => $image->getClientOriginalName()]);
                        $uploadImage = \Helpers::multipleUploadFiles($image, 'hotel/images');
                        if ($uploadImage['status'] == true) {
                            Hotel_Images::create([
                                'hotel_id' => $id,
                                'images' => $uploadImage['file_name'],
                            ]);
                        } else {
                            \Log::error('Image upload failed: ', ['error' => $uploadImage['message']]);
                        }
                    }
                }
                
                // // Handle additional info
                // HotelFeature::where('hotel_id',$id)->delete();
                // if(isset($feature_id_arr) && $feature_id_arr!=''){
                //         foreach ($feature_id_arr as $feature_id) {
                //             $data = array(
                //                 'hotel_id'=> $id,
                //                 'feature_id' => $feature_id
                //             );
                //             HotelFeature::create($data);
                //         }
                // }
                
                // Insert rules
                Hotel_Rules::where('hotel_id',$id)->delete();
                if(isset($rules_id_arr) && $rules_id_arr!=''){
                    foreach ($rules_id_arr as $rules_id) {
                        $data = array(
                            'hotel_id'=> $id,
                            'rules_id' => $rules_id
                        );
                        Hotel_Rules::create($data);
                    }
                }
                
                // Insert off days
                Hotel_Off_Days::where('hotel_id',$id)->delete();
                if(isset($off_days_arr) && $off_days_arr!=''){
                    foreach ($off_days_arr as $offDay) {
                        $data = array(
                            'hotel_id'=> $id,
                            'off_days' => $offDay
                        );
                        Hotel_Off_Days::create($data);
                    }
                }
    
                return ['status' => true, 'message' => __('lang.admin_data_update_msg')];
            }
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
        }
    }
    
    /**
     * Delete particular
    **/
    public static function deleteRecord($id) {
        try {
            $obj = new self;    
            $obj->where('id',$id)->delete();   
            return ['status' => true, 'message' => __('lang.admin_data_delete_msg')];
        }
        catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' '. $e->getLine() . ' '. $e->getFile()];
        } 
    }
    
     /**
     * Update Columns 
    **/
    public static function updateColumn($id){
        try {
            
            $data = self::where('id', $id)->first();
            
            $newStatus = ($data->status == '1') ? '0' : '1';
            
            $data->update(['status' => $newStatus]);
            
            return ['status' => true, 'message' => __('lang.admin_data_change_msg')];
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
        }
    }


}
