<?php

namespace App\Http\Controllers\API\Vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Auth;
use DB;
use Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\HttpFoundation\Response;
use App\Models\User;
use App\Models\Hotel;
use App\Models\Coupon;
use App\Models\CouponHotel;
use App\Models\CouponCustomer;


class VendorCouponAPIController extends Controller
{
    private $language;
    public function __construct()
    {
        // No need to call parent::__construct() here
        $this->language = request()->header('language-code') && request()->header('language-code') != '' ? request()->header('language-code') : 'en';
    }
    
    /**
     * Do Save Coupon
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function doSaveCoupon(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'added_type' => 'required|in:Vendor',
            'coupon_type' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'code' => 'required|string|max:50',
            'limit_for_same_user' => 'required|integer|min:1',
            'start_date' => 'required|date|date_format:Y-m-d',
            'expire_date' => 'required|date|after_or_equal:start_date|date_format:Y-m-d',
            'discount_type' => 'required|in:amount,percent',
            'discount' => 'required|numeric|min:0',
            'max_discount' => 'required|numeric|min:0',
            'hotel_id' => 'required|array',
            'customer_id' => 'required|array',
        ]);
    
        if ($validator->fails()) {
            return $this->sendError(__('lang.message_required_message'), $validator->errors());
        }
    
        DB::beginTransaction();
        try {
            if ($request->vendor) {
                $couponData = [
                    'added_type' => $request->added_type,
                    'vendor_id' => $request->vendor->id,
                    'coupon_type' => $request->coupon_type,
                    'title' => $request->title,
                    'code' => $request->code,
                    'limit_for_same_user' => $request->limit_for_same_user,
                    'start_date' => $request->start_date,
                    'expire_date' => $request->expire_date,
                    'discount_type' => $request->discount_type,
                    'discount' => $request->discount,
                    'max_discount' => $request->max_discount,
                    'created_at' => now(),
                ];
    
                // Insert Coupon Data
                $coupon_id = Coupon::insertGetId($couponData);
    
                $hotel_id_arr = $request->hotel_id;
                $customer_id_arr = $request->customer_id;
    
                // Insert into CouponHotel if not already exists
                if (!empty($hotel_id_arr)) {
                    foreach ($hotel_id_arr as $hotel_id) {
                        $exists = CouponHotel::where('hotel_id', $hotel_id)
                                             ->where('coupon_id', $coupon_id)
                                             ->exists();
                        if (!$exists) {
                            CouponHotel::create([
                                'hotel_id' => $hotel_id,
                                'coupon_id' => $coupon_id,
                            ]);
                        }
                    }
                }
    
                // Insert into CouponCustomer if not already exists
                if (!empty($customer_id_arr)) {
                    foreach ($customer_id_arr as $customer_id) {
                        $exists = CouponCustomer::where('customer_id', $customer_id)
                                                ->where('coupon_id', $coupon_id)
                                                ->exists();
                        if (!$exists) {
                            CouponCustomer::create([
                                'customer_id' => $customer_id,
                                'coupon_id' => $coupon_id,
                            ]);
                        }
                    }
                }
                $coupon = Coupon::with([
                    'vendor:id,name',
                    'hotels:id,name',
                    'customers:id,name',
                ])->find($coupon_id);
    
                DB::commit();
                return $this->sendResponse($coupon, __('lang.admin_data_add_msg'));
            } else {
                return $this->sendResponse([], __('lang.vendor_not_found'));
            }
        } catch (\Exception $ex) {
            DB::rollBack();
            \Log::error('Error saving coupon: ', ['exception' => $ex]);
            return $this->sendError($ex->getMessage());
        }
    }
    
    /**
    * Fetch Details here
    **/
    public function vendorCouponListing(Request $request)
    {
        try {
            $vendorId = $request->vendor->id;
            $couponData = Coupon::where('vendor_id', $vendorId)
                        ->where('vendor_id', '!=', 0)
                        ->with([
                            'vendor:id,name',
                            'hotels:id,name',
                            'customers:id,name',
                        ])
                        ->get();
    
            // Check if the collection is empty
            if ($couponData->isEmpty()) {
                return $this->sendError('No coupons found for this vendor');
            }
    
            // Return the response with the relevant data
            return $this->sendResponse($couponData, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            // Log the exception and return an error response
            \Log::error('Error fetching vendor coupons: ', ['exception' => $ex]);
            return $this->sendError('Error fetching data', $ex->getMessage());
        }
    }
    
    /**
    * Edit Details here
    **/
    public function vendorCouponEdit(Request $request)
    {
        try {
            // Validate the incoming request to ensure coupon_id is provided
            $validator = Validator::make($request->all(), [
                'coupon_id' => 'required|exists:coupons,id',
            ]);
    
            if ($validator->fails()) {
                return $this->sendError(__('lang.message_required_message'), $validator->errors());
            }
    
            // Fetch the vendor ID from the request
            $vendorId = $request->vendor->id;
    
            // Fetch the coupon data along with related vendor, hotels, and customers
            $couponData = Coupon::where('id', $request->coupon_id)
                ->with([
                    'vendor:id,name',
                    'hotels:id,name',
                    'customers:id,name',
                ])
                ->first();
    
            // Check if the coupon data is found
            if (!$couponData) {
                return $this->sendError(__('lang.coupon_not_found'));
            }
    
            // Return the response with the relevant data
            return $this->sendResponse($couponData, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            // Log the exception and return an error response
            \Log::error('Error fetching vendor coupon: ', ['exception' => $ex, 'coupon_id' => $request->coupon_id ?? 'N/A']);
            return $this->sendError('Error fetching data', $ex->getMessage());
        }
    }
    
    /**
     * Do Update Coupon
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function doUpdateCoupon(Request $request)
    {
        DB::beginTransaction();
        try {
            if ($request->vendor) {
                if (isset($request->couponId)) {
                    $couponId = $request->couponId;
                    $coupon = Coupon::find($couponId);
                    if (!$coupon) {
                        return $this->sendError('Coupon not found');
                    }
                } else {
                    return $this->sendError('Coupon ID is required');
                }
                $couponData = [
                    'updated_at' => now(),
                ];

                $fields = [
                    'coupon_type', 'title', 'code', 'limit_for_same_user',
                    'start_date', 'expire_date', 'discount_type', 'discount',
                    'max_discount',
                ];
                foreach ($fields as $field) {
                    if ($request->has($field)) {
                        $couponData[$field] = $request->$field;
                    }
                }
                $coupon->update($couponData);
                $hotelIdArr = $request->hotel_id ?? [];
                if (!empty($hotelIdArr)) {
                    CouponHotel::where('coupon_id', $couponId)->delete();
                    foreach ($hotelIdArr as $hotelId) {
                        CouponHotel::firstOrCreate([
                            'hotel_id' => $hotelId,
                            'coupon_id' => $couponId,
                        ]);
                    }
                }
                $customerIdArr = $request->customer_id ?? [];
                if (!empty($customerIdArr)) {
                    CouponCustomer::where('coupon_id', $couponId)->delete();
                    foreach ($customerIdArr as $customerId) {
                        CouponCustomer::firstOrCreate([
                            'customer_id' => $customerId,
                            'coupon_id' => $couponId,
                        ]);
                    }
                }
                $coupon = Coupon::with([
                    'vendor:id,name',
                    'hotels:id,name',
                    'customers:id,name',
                ])->find($couponId);
    
                DB::commit();
                return $this->sendResponse($coupon, __('lang.admin_data_add_msg'));
            } else {
                return $this->sendResponse([], __('lang.vendor_not_found'));
            }
        } catch (\Exception $ex) {
            DB::rollBack();
            \Log::error('Error updating coupon: ', ['exception' => $ex, 'coupon_id' => $request->couponId ?? 'N/A']);
            return $this->sendError('Error updating coupon', $ex->getMessage());
        }
    }
    
    /**
    * Fetch Details here
    **/
    public function hotelListing(Request $request)
    {
        try {
            $hotelData = Hotel::where('status', 1)
                ->where('vendor_id',$request->vendor->id)
                ->select('id','name')
                ->get();
    
            return $this->sendResponse($hotelData, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }
    
    /**
    * Fetch Details here
    **/
    public function customerListing(Request $request)
    {
        try {
            $userData = User::where('status', 1)
                ->where('type','user')
                ->where('role_id','2')
                ->whereNotNull('name')
                ->where('name', '!=', '')
                ->select('id','name')
                ->get();
    
            return $this->sendResponse($userData, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }
    
    
    
    
}
