<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\RecentlyViewedHotel;
use Illuminate\Support\Str;
use Auth;
use Validator;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use App\Models\Hotel;
use App\Models\Additional_Info;
use App\Models\Destination;
use App\Models\Feature;
use App\Models\Rules;
use App\Models\Cancellation_Policy;
use App\Models\Specification;
use App\Models\Hotel_Aminities;
use App\Models\Hotel_Images;
use Session;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\HttpFoundation\Response;

class HotelAPIController extends Controller
{
    private $language;
    public function __construct()
    {
        // No need to call parent::__construct() here
        $this->language = request()->header('language-code') && request()->header('language-code') != '' ? request()->header('language-code') : 'en';
    }

    /**
     * Get List
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getLatestHotel(Request $request)
    {
        try {
            $pagination_no = config('constant.api_pagination');
            if (isset($request->per_page) && !empty($request->per_page)) {
                $pagination_no = $request->per_page;
            }

            // Fetch hotels with related category and destination
            $hotels = Hotel::where('status', 1)
                ->with(['destination'])
                ->select('id', 'name', 'price', 'type', 'destination_id', 'front_image')
                ->latest()->take(4)->get(); // Adjust pagination if needed

            // Process the retrieved data
            foreach ($hotels as $hotel) {
                $hotel->front_image = url('uploads/hotel/front_image/' . $hotel->front_image);
                // Add destination name to the response
                $hotel->destination_name = $hotel->destination ? $hotel->destination->name : null;
            }

            // Send response
            $response = $this->sendResponse($hotels, __('lang.message_data_retrived_successfully'));
            return $response;
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }
    
    /**
     * Fetch list here
    **/
    public  function getLocationList(Request $request){
        try{
            $controller = new self(); // Creating an instance of HomeAPIController
            $pagination_no = config('constant.api_pagination');
            if(isset($search['per_page']) && !empty($search['per_page'])){
                $pagination_no = $search['per_page'];
            }
            $resData = Destination::where('status',1)->select('id','name')->get();
            if(count($resData)){
                foreach($resData as $row){
                    $row->image = url('uploads/destination/'.$row->image) ?? null;
                }
            }
            
            $response = $controller->sendResponse($resData, __('lang.message_data_retrived_successfully'));  
            return $response;
        } catch (\Exception $ex) {
            return $controller->sendError($ex->getMessage());
        }
    }
    
    /**
     * Fetch list here
    **/
    public  function getFeatureList(Request $request){
        try{
            $controller = new self(); // Creating an instance of HomeAPIController
            $pagination_no = config('constant.api_pagination');
            if(isset($search['per_page']) && !empty($search['per_page'])){
                $pagination_no = $search['per_page'];
            }
            $resData = Feature::where('status',1)->select('id','name')->get();
            
            $response = $controller->sendResponse($resData, __('lang.message_data_retrived_successfully'));  
            return $response;
        } catch (\Exception $ex) {
            return $controller->sendError($ex->getMessage());
        }
    }
    
    /**
     * Get Full List
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getHotelList(Request $request)
    {
        try {
            // Set default pagination
            $pagination_no = config('constant.api_pagination', 10);
    
            // Update pagination if per_page is set
            if ($request->has('per_page') && !empty($request->per_page)) {
                $pagination_no = (int) $request->per_page;
            }
    
            // Start building the query
            $query = Hotel::where('status', 1)
                ->with(['destination:id,name', 'aminities:id,name', 'ratings'])
                ->select('id', 'name', 'price', 'type', 'destination_id', 'front_image');
    
            // Apply destination filter
            if ($request->has('destination_id') && !empty($request->destination_id)) {
                $query->where('destination_id', $request->destination_id);
            }
    
            // Apply price filter
            if ($request->has('price') && !empty($request->price)) {
                $priceRange = explode('-', $request->price);
                if (count($priceRange) == 2) {
                    $query->whereBetween('price', [$priceRange[0], $priceRange[1]]);
                } else {
                    // Handle single value price filter
                    $query->where('price', $priceRange[0]);
                }
            }
    
            // Get the results with pagination
            $hotels = $query->latest()->paginate($pagination_no);
    
            // Process each hotel to add required data
            $hotels->transform(function ($hotel) {
                // Build the front image URL
                $hotel->front_image = url('uploads/hotel/front_image/' . $hotel->front_image);
                
                // Add destination name
                $hotel->destination_name = $hotel->destination ? $hotel->destination->name : null;
                
                // Add amenities list
                $hotel->amenities_list = $hotel->aminities->pluck('name');
                
        
               // Calculate and add average rating
                $hotel->average_rating = $hotel->ratings->avg('rating') ?? null;
                unset($hotel->ratings);
                
                
                // Add destination id and name separately
                if ($hotel->destination) {
                    $hotel->destination = [
                        'id' => $hotel->destination->id,
                        'name' => $hotel->destination->name
                    ];
                }
                
                return $hotel;
            });
    
            // Send response
            return $this->sendResponse($hotels, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            // Send error response
            return $this->sendError($ex->getMessage());
        }
    }
    
    /**
     * Fetch Details here
    **/
    public function hotelDetails(Request $request)
    {
        try {
            
            $hotelData = Hotel::where('status', 1)
                ->with('destination')
                ->where('id', $request->hotelId)
                ->first();
    
            if (!$hotelData) {
                return $this->sendError('Hotel not found');
            }
    
            // Format the response using the helper function
            $formattedHotelData = \Helpers::formatHotelData($hotelData);
    
            // Return the response with the relevant data
            return $this->sendResponse($formattedHotelData, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            // Return error response in case of exception
            return $this->sendError($ex->getMessage());
        }
    }

    
    /**
     * Get Recommended List
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getRecommendedList(Request $request)
    {
        try {
            // Set default pagination
            $pagination_no = config('constant.api_pagination', 10);
    
            // Update pagination if per_page is set
            if ($request->has('per_page') && !empty($request->per_page)) {
                $pagination_no = (int) $request->per_page;
            }
    
            // Start building the query
            $query = Hotel::where('status', 1)
                ->where('is_recommended','Yes')
                ->with(['destination:id,name', 'aminities:id,name'])
                ->select('id', 'name', 'price', 'type', 'destination_id', 'front_image');
    
            // Get the results with pagination
            $hotels = $query->latest()->get();
    
            // Process each hotel to add required data
            $hotels->transform(function ($hotel) {
                // Build the front image URL
                $hotel->front_image = url('uploads/hotel/front_image/' . $hotel->front_image);
                
                // Add destination name
                $hotel->destination_name = $hotel->destination ? $hotel->destination->name : null;
                
                // Add amenities list
                $hotel->amenities_list = $hotel->aminities->pluck('name');
                
                // Add destination id and name separately
                if ($hotel->destination) {
                    $hotel->destination = [
                        'id' => $hotel->destination->id,
                        'name' => $hotel->destination->name
                    ];
                }
                
                return $hotel;
            });
    
            // Send response
            return $this->sendResponse($hotels, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            // Send error response
            return $this->sendError($ex->getMessage());
        }
    }
    
    /**
     * Get Recommended List
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getExplorePlace(Request $request)
    {
        try {
            // Set default pagination
            $pagination_no = config('constant.api_pagination', 10);
    
            // Update pagination if per_page is set
            if ($request->has('per_page') && !empty($request->per_page)) {
                $pagination_no = (int) $request->per_page;
            }
    
            // Get the destinations ordered by the number of bookings
            $explorePlace = Destination::select('destinations.*')
                ->join('hotels', 'hotels.destination_id', '=', 'destinations.id')
                ->join('bookings', 'bookings.hotel_id', '=', 'hotels.id')
                ->where('destinations.status', 1)
                ->groupBy('destinations.id')
                ->orderByRaw('COUNT(bookings.id) DESC')
                ->get();
            if(count($explorePlace)){
                    foreach($explorePlace as $row){
                        $row->image = url('uploads/destination/'.$row->image) ?? null;
                    }
                }
    
            // Send response
            return $this->sendResponse($explorePlace, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            // Send error response
            return $this->sendError($ex->getMessage());
        }
    }
    
    /**
     * Fetch list here
    **/
    public  function getRulesList(Request $request){
        try{
            $controller = new self(); // Creating an instance of HomeAPIController
            $pagination_no = config('constant.api_pagination');
            if(isset($search['per_page']) && !empty($search['per_page'])){
                $pagination_no = $search['per_page'];
            }
            $resData = Rules::where('status',1)->select('id','name')->get();
            
            $response = $controller->sendResponse($resData, __('lang.message_data_retrived_successfully'));  
            return $response;
        } catch (\Exception $ex) {
            return $controller->sendError($ex->getMessage());
        }
    }
    
    /**
     * Save View Hotel
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function saveViewHotel(Request $request)
    {
        try {
            $hotelId = $request->hotelId;
            $userId = $request->userId ?: 0;
            $deviceToken = $request->header('device-token');
            Log::info('Device Token:', ['device_token' => $deviceToken]);
            if (!$deviceToken) {
                return $this->sendError(__('lang.device_token_required'));
            }
            $postData = [
                'user_id' => $userId,
                'device_token' => $deviceToken,
                'hotel_id' => $hotelId,
            ];
            $recentlyViewedHotel = RecentlyViewedHotel::create($postData);
            return $this->sendResponse($recentlyViewedHotel->id, __('lang.admin_data_add_msg'));
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }
    
    /**
     * Recently View List
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getRecentlyView(Request $request)
    {
        try {
            $pagination_no = config('constant.api_pagination', 10);
            if ($request->has('per_page') && !empty($request->per_page)) {
                $pagination_no = (int) $request->per_page;
            }
            $userId = $request->userId;
            $sessionId = Session::getId();
            $query = RecentlyViewedHotel::query()
                ->where(function ($query) use ($userId, $sessionId) {
                    if ($userId) {
                        $query->where('user_id', $userId);
                    } else {
                        $query->where('session_id', $sessionId);
                    }
                })
                ->orderBy('created_at', 'desc')
                ->pluck('hotel_id')
                ->map(function ($id) {
                    return (int) $id;
                });
            // Fetch hotel details
            $hotels = Hotel::whereIn('id', $query->toArray())
                // ->with(['destination:id,name', 'aminities:id,name'])
                // ->select('id', 'name', 'price', 'type', 'destination_id', 'category_id', 'front_image')
                ->get();
            $hotels->transform(function ($hotel) {
                $hotel->front_image = url('uploads/hotel/front_image/' . $hotel->front_image);
                $hotel->destination_name = $hotel->destination ? $hotel->destination->name : null;
                $hotel->amenities_list = $hotel->aminities->pluck('name');
                if ($hotel->destination) {
                    $hotel->destination = [
                        'id' => $hotel->destination->id,
                        'name' => $hotel->destination->name
                    ];
                }
                return $hotel;
            });
            return $this->sendResponse($hotels, __('lang.message_data_retrived_successfully'));
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }




}
