<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Auth;
use DB;
use Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\HttpFoundation\Response;
use App\Models\User;
use App\Models\Booking;
use App\Models\BookingGuest;
use App\Models\Package;
use App\Models\Coupon;
use App\Models\Hotel_Packages;
use App\Models\CouponHotel;
use App\Models\CouponCustomer;

class CouponAPIController extends Controller
{
    private $language;
    public function __construct()
    {
        // No need to call parent::__construct() here
        $this->language = request()->header('language-code') && request()->header('language-code') != '' ? request()->header('language-code') : 'en';
    }

        /**
     * Do Apply Coupon Code
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function doApplyCoupon(Request $request)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'booking_id' => 'required|exists:bookings,id',
            'code' => 'required|exists:coupons,code',
        ]);
    
        if ($validator->fails()) {
            return $this->sendError(__('lang.message_required_message'), $validator->errors());
        }
    
        try {
            $customer_id = $request->user->id;
            $booking_id = $request->booking_id;
            $code = $request->code;
    
            // Fetch the booking and hotel_id
            $booking = Booking::findOrFail($booking_id);
            $hotel_id = $booking->hotel_id;
    
            // Fetch the coupon
            $coupon = Coupon::where('code', $code)->firstOrFail();
    
            // Check if the coupon is valid by date and status
            $currentDate = Carbon::now();
            $startDate = Carbon::parse($coupon->start_date);
            $expireDate = Carbon::parse($coupon->expire_date)->endOfDay();
    
            if ($currentDate->lt($startDate)) {
                return $this->sendError(__('lang.message_coupon_not_active_yet'));
            }
    
            if ($currentDate->gt($expireDate)) {
                return $this->sendError(__('lang.message_coupon_expired'));
            }
    
            if ($coupon->status != '1') {
                return $this->sendError(__('lang.message_coupon_invalid'));
            }
    
            // Check if the coupon is valid for the hotel
            $couponHotel = CouponHotel::where('coupon_id', $coupon->id)
                                      ->where('hotel_id', $hotel_id)
                                      ->first();
            if (!$couponHotel) {
                return $this->sendError(__('lang.message_coupon_not_valid_for_hotel'));
            }
    
            // Check if the coupon is valid for the user
            $couponCustomer = CouponCustomer::where('coupon_id', $coupon->id)
                                            ->where('customer_id', $customer_id)
                                            ->first();
            if (!$couponCustomer) {
                return $this->sendError(__('lang.message_coupon_not_valid_for_user'));
            }
    
            // Check the coupon usage limit
            $usageCount = CouponCustomer::where('coupon_id', $coupon->id)
                                        ->where('customer_id', $customer_id)
                                        ->count();
            if ($usageCount >= $coupon->limit_for_same_user) {
                return $this->sendError(__('lang.message_coupon_limit_exceeded'));
            }
    
            // Calculate the discount
            $originalPrice = $booking->price;
            $discount = 0;
    
            if ($coupon->discount_type == 'amount') {
                $discount = min($coupon->discount, $coupon->max_discount);
            } else if ($coupon->discount_type == 'percent') {
                $discount = ($originalPrice * $coupon->discount / 100);
                $discount = min($discount, $coupon->max_discount);
            }
    
            // Check if the price after discount meets the minimum purchase requirement
            $finalPrice = $originalPrice - $discount;
            if ($finalPrice < $coupon->min_purchase) {
                return $this->sendError(__('lang.message_price_below_minimum_purchase'));
            }
    
            // Update the booking with the discounted price
            if ($discount > 0) {
                DB::transaction(function () use ($booking, $finalPrice, $discount) {
                    $booking->update([
                        'price' => $finalPrice,
                        'discount' => $discount
                    ]);
                });
            }
    
            // Return the discounted price
            return $this->sendResponse([
                'original_price' => $originalPrice,
                'final_price' => $finalPrice,
                'discount' => $discount
            ], __('lang.message_coupon_applied_successfully'));
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }
}
